/*  Prüft die Einträge der Tabelle "scheduling.resource_timeline" vom Typ "task" auf logische Inkonsistenzen.
    Diese Funktion wird üblicherweise als Unterfunktion der DB-Funktion "scheduling.resource_timeline__validate_block" gerufen.
*/
SELECT tsystem.function__drop_by_regex( 'resource_timeline__validate_block_task', 'scheduling', _commit => true );
CREATE OR REPLACE FUNCTION scheduling.resource_timeline__validate_block_task(
    _block scheduling.resource_timeline
) RETURNS jsonb AS $$
DECLARE
    _ksvba_id int := context_id FROM scheduling.resource WHERE id = _block.ti_resource_id and context = 'ksvba';

    _issues jsonb := '{ "has_issues": false, "issues":[] }';

    _max_usage numeric := 0.;
BEGIN
    -- Sicher gehen, dass hier wirklich ein Eintrag vom Typ "task" validiert wird.
    IF ( _block.ti_type <> 'task' ) THEN
        RAISE EXCEPTION 'invalid block type, id % is not of type task', _block.ti_id;
    END IF;

    -- Ein Task-Eintrag hat immer eine Belegung größer 0 und kleiner gleich 1.
    IF NOT( _block.ti_usage <@ '(0,1]'::numrange ) THEN
        _issues := jsonb_set( _issues, '{has_issues}', 'true' );
        _issues := jsonb_insert( _issues, '{ issues, 0 }', '"load outside allowed range"' );
    END IF;

    -- Der Beginn des Task-Eintrag liegt nie vor dem Start der Arbeitszeit.
    IF (
      _block.ti_date_start::time < (scheduling.ksvba__get_shifttimes( _ksvba_id ))[1]
    ) THEN
        _issues := jsonb_set( _issues, '{has_issues}', 'true' );
        _issues := jsonb_insert( _issues, '{ issues, 0 }', '"start before resource shift hours"' );
    END IF;

    -- Das Ende des Task-Eintrag liegt nie nach dem Ende der Arbeitszeit.
    IF (
      _block.ti_date_end::time > (scheduling.ksvba__get_shifttimes( _ksvba_id ))[2]
    ) THEN
        _issues := jsonb_set( _issues, '{has_issues}', 'true' );
        _issues := jsonb_insert( _issues, '{ issues, 0 }', '"end after resource shift hours"' );
    END IF;

    -- Es kann keinen Task-Eintrag geben, wenn die referenzierte Ressource gesperrt ist.
    IF
        k.ks_sperr
        FROM ksv k
        JOIN ksvba ba ON k.ks_id = ba.ksb_ks_id
        WHERE ba.ksb_id = _ksvba_id
    THEN
        _issues := jsonb_set( _issues, '{has_issues}', 'true' );
        _issues := jsonb_insert( _issues, '{ issues, 0 }', '"resource locked"' );
    END IF;

    -- Der Task-Eintrag liegt nie an einem arbeitsfreien Tag.
    IF (
          extract( dow from _block.ti_date_start )
        <> all( scheduling.ksvba__get_workingdays( _ksvba_id ) )
    ) THEN
        _issues := jsonb_set( _issues, '{has_issues}', 'true' );
        _issues := jsonb_insert( _issues, '{ issues, 0 }', '"not a working day"' );
    END IF;

    -- Die maximal Belegungen der Ressource darf den Wert von 1 im Zeitinterval des aktuellen Task-Eintrags nicht übersteigen.
    -- Dafür wird zu jedem Startzeitpunkt eines Task-Blocks innerhalb des Zeitintervals des gegebenen Task-Eintrags geschaut,
    -- wie die Gesamtbelastung der Ressouce ist. Task-Einträge die zu dem entsprechenden Zeitpunkten enden, werden ignoriert,
    -- damit es bei gleichzeitig startenden und endenden Blöcken nicht zu künstlichen Überlastungen kommt.
    WITH
        --- Die Task-Einträge im Zeitinterval des aktuellen Task-Eintrags.
        _raw_data AS (
            SELECT *
            FROM scheduling.resource_timeline
            WHERE tsrange( _block.ti_date_start, _block.ti_date_end, '[]' ) && tsrange( ti_date_start, ti_date_end, '[]' )
            AND ti_type IN ( 'task', 'task.buffer', 'task.blocktime' )
            AND ti_resource_id = _block.ti_resource_id
        ),
        --- Alle Start-Zeiten der Task-Einträge im Zeitinterval des aktuellen Task-Eintrags. Nur bei den Start-Zeiten kann sich die Belegung der Ressource erhöhen.
        _times AS (
            SELECT DISTINCT ti_date_start AS _point_in_time
            FROM _raw_data
            WHERE ti_date_start <@ tsrange( _block.ti_date_start, _block.ti_date_end, '[]' )
        ),
        --- Die Gesamtbelegung der Ressource zu den vorher ermittelten Zeitpunkten.
        _usage_at_times AS (
            SELECT
              t._point_in_time,
              sum( rd.ti_usage ) AS _total_usage
            FROM _times t
            JOIN _raw_data rd ON t._point_in_time <@ tsrange( rd.ti_date_start, rd.ti_date_end, '[)' ) --
            GROUP BY t._point_in_time
        )
    --- Maximale Belegung der Ressource im Zeitinterval des aktuellen Task-Eintrags.
    SELECT max( _total_usage )
    INTO _max_usage
    FROM _usage_at_times;
     IF ( _max_usage > 1.0 ) THEN
        _issues := jsonb_set( _issues, '{has_issues}', 'true' );
        _issues := jsonb_insert( _issues, '{ issues, 0 }', '"maximal load of resource exceeded"' );
    END IF;

    -- AG ist als erledigt markiert, aber noch terminiert.
    IF (
            ( SELECT a2_ende FROM ab2 WHERE a2_id = _block.ti_a2_id )
        AND EXISTS(
          SELECT true
          FROM scheduling.resource_timeline
          WHERE
                ti_id = _block.ti_id
            AND ti_type = 'task'
        )
    ) THEN
        _issues := jsonb_set( _issues, '{has_issues}', 'true' );
        _issues := jsonb_insert( _issues, '{ issues, 0 }', '"marked as done but still present in timeline"'::jsonb );
    END IF;

    -- raise exception '%', _issues;

    RETURN _issues;

END $$ language plpgsql;
